import logging
import os
import sys
import time
from selenium import webdriver
from selenium.webdriver import Remote
from selenium.webdriver.common.by import By
from selenium.webdriver.support.ui import WebDriverWait, Select
from selenium.webdriver.support import expected_conditions as EC
from webdriver_manager.chrome import ChromeDriverManager
from selenium.webdriver.chrome.service import Service

from pipeline.email_manager.gmail_api import (
    init_gmail_service,
    get_email_message_details,
    search_emails,
)
from bs4 import BeautifulSoup


def create_browser_window():
    test_ua = "Mozilla/5.0 (Windows NT 4.0; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/130.0.0.0 Safari/537.36"
    options = webdriver.ChromeOptions()
    options.add_argument("--lang=pt-BR")
    options.add_argument(f"--user-agent={test_ua}")
    options.add_argument('--no-sandbox')
    options.add_argument('--disable-dev-shm-usage')

    browser = Remote(
        command_executor="http://chrome:4444/wd/hub",
        desired_capabilities={"browserName": "chrome"},
        options=options,
    )

    return browser


def verify_on_chrome(url: str):
    driver = create_browser_window()
    driver.get(url)

    try:
        logging.warning(f"verify_on_chrome....")
        file_input = WebDriverWait(driver, 10).until(
            EC.presence_of_element_located((By.ID, "friendlyName"))
        )
        file_input.clear()
        time.sleep(5)
        file_input.send_keys("Roku Express")

        logging.warning(f"verify_on_chrome - roku name = Roku Express....")

        time.sleep(5)

        dropdown_element = WebDriverWait(driver, 10).until(
            EC.presence_of_element_located((By.ID, "name"))
        )

        dropdown = Select(dropdown_element)
        dropdown.select_by_visible_text("Sala de estar")

        logging.warning(f"verify_on_chrome - roku place = Sala de estar....")
        time.sleep(5)

        driver.execute_script("document.getElementById('EULA').click()")

        logging.warning(f"verify_on_chrome - EULA Check....")
        time.sleep(5)

        continuar_button = WebDriverWait(driver, 10).until(
            EC.element_to_be_clickable(
                (
                    By.XPATH,
                    "//div[@class='ctaButtonContainer']/button[@aria-label='Continuar']",
                )
            )
        )
        logging.warning(f"verify_on_chrome - Continuar clicked....")
        continuar_button.click()

        time.sleep(30)

        continuar_button = WebDriverWait(driver, 10).until(
            EC.element_to_be_clickable(
                (
                    By.XPATH,
                    "//button[contains(@class, 'roku-button') and contains(text(), 'Continuar')]",
                )
            )
        )
        logging.warning(f"verify_on_chrome - Continuar clicked....")
        continuar_button.click()

        time.sleep(30)
        return True

    except Exception as e:
        logging.warning(f"An error occurred... {e}")
        driver.quit()
        return False


def _ask_for_url(email_recipient: str):
    client_file = "pipeline/email_manager/client_secret.json"
    logging.warning(f"Starting gmail service....")
    service = init_gmail_service(client_file)

    urls = []

    query = f"from:('Roku' via) to:('{email_recipient}') subject:(ativar seu dispositivo)"
    logging.warning(f"Searching for emails....")
    email_messages = search_emails(service, query, max_results=5)

    for message in email_messages:
        detail = get_email_message_details(service, message["id"])

        soup = BeautifulSoup(detail["body"], features="html.parser")
        element = soup.find("table").find("a")
        element = soup.find("table").find("a")
        logging.warning(f"\nSubject: {detail["subject"]}")
        logging.warning(f"Sender: {detail["sender"]}")
        logging.warning(f"Date: {detail["date"]}")
        logging.warning(f"Recipients: {detail["recipients"]}")
        urls.append(element["href"], )

    return urls


def main():
    email_recipient = os.getenv('EMAIL_CONFIRMATION_RECIPIENT', 'roku.dev@cdn.tv.br')
    confirmed = False
    
    try:
        logging.warning(f"Starting pipeline....")
        logging.warning(f"email_recipient = {email_recipient}")
        urls = _ask_for_url(email_recipient)
        for url in urls:
            logging.warning(f"\nURL: {url}\n")
            confirmed = confirmed or verify_on_chrome(url=url)

        if confirmed:
            sys.exit(0)
        sys.exit(1)
    except KeyboardInterrupt:
        sys.exit(1)


if __name__ == "__main__":
    main()