import logging
import os
import sys
import time

import requests
from skimage.metrics import structural_similarity as ssim
import shutil
from bs4 import BeautifulSoup
import cv2
from requests.auth import HTTPDigestAuth
import pytesseract
from PIL import Image

class RaisedException(Exception):
    pass

def call_remote_control_api(action):
    status_code = 0

    while status_code != 200:
        url = f"http://api.remote.roku.cdnsystem.cloud/{action}"

        payload = {}
        headers = {}

        response = requests.request("POST", url, headers=headers, data=payload)
        status_code = response.status_code

        logging.warning(status_code)
        logging.warning(response.text)
        if status_code != 200:
            logging.warning(f"Error on remote control API. action: {action}")

            time.sleep(2)

def _identify_screen(images_data_list, frame_bgr2gray):
    for image_data in images_data_list:
        # time.sleep(1)
        area_to_compare_croped = frame_bgr2gray[
                image_data['dimension_a']:image_data['dimension_b'],
                image_data['dimension_c']:image_data['dimension_d']
            ]

        similarity, _ = ssim(image_data['reference_image'], area_to_compare_croped, full=True)

        # cv2.imwrite('frame_croped.png', area_to_compare_croped)
        # cv2.imwrite('reference_image.png', image_data['reference_image'])
        if similarity > 0.9:
            # cv2.imwrite('frame_croped_match.png', area_to_compare_croped)
            # cv2.imwrite('reference_image_match.png', image_data['reference_image'])
            logging.warning(f"Similarity: {similarity}")
            logging.warning("Matching Area Found...")
            return image_data['name']


def capturar_e_comparar(images_data_list, func_action):
    cap = _get_video_capture()

    cap.set(cv2.CAP_PROP_BUFFERSIZE, 0)
    call_remote_control_api('rev')
    call_remote_control_api('home')

    # To check if the job started with Roku signed in to another account
    attemps = 0
    while True:
        for _ in range(10):
            cap.grab()
        ret, frame = cap.read()
        if not ret:
            cap.release()
            raise RaisedException("Erro ao capturar o frame")

        # logging.warning('New Frame')
        frame_bgr2gray = cv2.cvtColor(frame, cv2.COLOR_BGR2GRAY)
        # cv2.imwrite('print.png', frame_bgr2gray)

        screen =  _identify_screen(images_data_list, frame_bgr2gray)
        attemps = func_action(
            screen, cap, attemps
        )

        attemps += 1
        time.sleep(3)

def _get_video_capture():
    connected = False
    input_index = 0

    while not connected:
        try:
            input_video_capture = input_index
            cap = cv2.VideoCapture(input_video_capture)
            connected = cap.isOpened()
        except Exception as e:
            logging.warning("Except on get Video Input")
            logging.warning(e)

        input_index += 1

        if input_index > 10:
            logging.warning("Error on get Video Input")
            exit(1)
    return cap

def get_number(frame_bgr2gray):
    area_to_read = frame_bgr2gray[340:360, 448:478]
    _, image_processed = cv2.threshold(area_to_read, 0, 255, cv2.THRESH_BINARY + cv2.THRESH_OTSU)
    number = pytesseract.image_to_string(image_processed, config="--psm 7 digits")
    number = number.strip()
    logging.warning(f"Número encontrado: {number}")
    return number


def get_dimensions(file_name):
    file_name = file_name.replace('.png', '')
    parts = file_name.split('_')

    if len(parts) != 5:
        logging.warning("Wrong format file name")
        exit(1)

    return parts[0], int(parts[1]), int(parts[2]), int(parts[3]), int(parts[4])


def get_all_images_frame():
    images_data_list = []
    filelist=os.listdir('images')
    for fichier in filelist[:]:
        if not(fichier.endswith(".png")):
            continue

        name, *dimensions = get_dimensions(fichier)
        reference_image = cv2.imread(f'images/{fichier}', cv2.IMREAD_GRAYSCALE)

        images_data_list.append({
            'file': fichier,
            'name': name,
            'dimension_a': dimensions[0],
            'dimension_b': dimensions[1],
            'dimension_c': dimensions[2],
            'dimension_d': dimensions[3],
            'reference_image': reference_image,
        })

    return images_data_list

def _baixar_imagem_logo(url_base):
    url_logo = f"{url_base}/logo/main-logo.png"
    caminho_imagem = 'main-logo.png'

    try:
        response = requests.get(url_logo)
        response.raise_for_status()
        with open(caminho_imagem, "wb") as file:
            file.write(response.content)
        logging.warning(f"Imagem 'main-logo.png' baixada de: {url_logo}")
        return caminho_imagem
    except Exception as e:
        logging.warning(f"Erro ao baixar a imagem 'main-logo.png': {e}")
        return None


def _redimensionar_imagem(caminho_imagem, tamanho=(540, 405)):
    try:
        imagem = Image.open(caminho_imagem)
        imagem = imagem.resize(tamanho)
        imagem.save(caminho_imagem)
        logging.warning(f"Imagem redimensionada para {tamanho}")
    except Exception as e:
        logging.warning(f"Erro ao redimensionar a imagem: {e}")


def get_logo(manager_url):
    if caminho_imagem := _baixar_imagem_logo(
        manager_url
    ):
        _redimensionar_imagem(caminho_imagem)


def enviar_instalar_roku(ip_roku, zip_file_path):
    try:
        url = f'http://{ip_roku}/plugin_install'
        files = {'archive': open(zip_file_path, 'rb')}
        data = {
            'command': 'install',
            'mysubmit': 'Install'
        }
        response = requests.post(
            url,
            files=files,
            data=data,
            auth=HTTPDigestAuth('rokudev', 'rokudev'),
            headers={'Origin': f'http://{ip_roku}'},
        )
        logging.warning(f'Status Code: {response.status_code}')
            #logging.warning(response.text)

    except Exception as e:
        logging.warning(f"Erro ao enviar arquivo para o Roku: {e}")
        return None


def converter_para_squashfs(ip_roku, zip_file_path):
    try:
        url = f'http://{ip_roku}/plugin_install'

        headers = {
            "Accept": "text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8,application/signed-exchange;v=b3;q=0.7",
            "Accept-Encoding": "gzip, deflate",
            "Accept-Language": "pt-BR,pt;q=0.9,en-US;q=0.8,en;q=0.7",
            "Cache-Control": "max-age=0",
            "Connection": "keep-alive",
            "Origin": f"http://{ip_roku}",
            "Referer": f"http://{ip_roku}/plugin_install",
            "Upgrade-Insecure-Requests": "1",
            "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/130.0.0.0 Safari/537.36"
        }

        files = {'archive': open(zip_file_path, 'rb')}
        data = {
            'command': 'convert_to_squashfs',
            'mysubmit': 'Convert to squashfs'
        }

        try:
            response = requests.post(url, headers=headers, files=files, data=data, auth=HTTPDigestAuth('rokudev', 'rokudev'))
            logging.warning(f'Status Code: {response.status_code}')
            #logging.warning(response.text)
        except Exception as e:
            logging.warning(f"Erro ao enviar a requisição: {e}")
    except Exception as e:
        logging.warning(f"Erro ao realizar a conversão: {e}")
        return False


def fazer_package(ip_roku, pkg_password):
    try:
        client_version = os.getenv('CLIENT_VERSION', '')

        url = f'http://{ip_roku}/plugin_package'
        channel_name = os.getenv('CHANNEL_NAME', '')

        data = {
            'mysubmit': 'Package',
            'pkg_time': '1730202899503',
            'app_name': f'{channel_name}/{client_version}',
            'passwd': f'{pkg_password}'
        }

        headers = {
            "Accept": "text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8,application/signed-exchange;v=b3;q=0.7",
            "Accept-Encoding": "gzip, deflate",
            "Accept-Language": "pt-BR,pt;q=0.9,en-US;q=0.8,en;q=0.7",
            "Connection": "keep-alive",
            "Host": ip_roku,
            "Referer": f"http://{ip_roku}/plugin_install",
            "Upgrade-Insecure-Requests": "1",
            "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/130.0.0.0 Safari/537.36"
        }

        try:
            response = requests.post(url, headers=headers, files=data, auth=HTTPDigestAuth('rokudev', 'rokudev'))
            logging.warning(f'Status Code: {response.status_code}')

            html_content = response.text
            soup = BeautifulSoup(html_content, "html.parser")
            if pkg_link := soup.find(
                "a", href=True, string=lambda x: x and x.endswith(".pkg")
            ):
                download_url = f"http://{ip_roku}/{pkg_link['href'].lstrip('/')}"
                pkg_filename = pkg_link.text

                with requests.get(download_url, stream=True, auth=HTTPDigestAuth('rokudev', 'rokudev')) as pkg_response:
                    pkg_response.raise_for_status()
                    with open(pkg_filename, "wb") as file:
                        for chunk in pkg_response.iter_content(chunk_size=8192):
                            file.write(chunk)

                shutil.copyfile(pkg_filename, 'new.pkg')
                logging.warning(f"Download concluído: {pkg_filename}")
            else:
                logging.warning("Arquivo .pkg não encontrado na resposta HTML.")
                logging.warning("Durante os testes, esse erro ocorria quando a senha do PKG estava errada")
                # logging.warning(response.text)
                exit(1)
        except Exception as e:
            logging.warning(f"Erro ao enviar a requisição: {e}")
            exit(1)
    except Exception as e:
        logging.warning("Erro")
        exit(1)

# def _get_email(frame_bgr2gray):
#     area_to_read = frame_bgr2gray[150:180, 40:450]
#     _, image_processed = cv2.threshold(area_to_read, 0, 255, cv2.THRESH_BINARY + cv2.THRESH_OTSU)
#     email = pytesseract.image_to_string(image_processed, config="--psm 7 digits")
#     logging.warning("Email encontrado:", email)
#     email = email.strip()
#     cv2.imwrite('email_image.png', area_to_read)
#     logging.warning("Email encontrado:", email)
#     return email
